﻿// See https://aka.ms/new-console-template for more information
using SerienHelper.API;
using SerienHelper.API.Enum;
using SerienHelper.API.Interfaces;
using System.CommandLine;
using System.Reflection;
using System.Text;
using System.Text.Json;
using System.Text.RegularExpressions;
using static SerienHelper.API.Handling.BaseWebsiteHandler;
using System.ComponentModel;

namespace SerienHelper;

class Program
{
	private static string HeaderText = "" +
		"********************************************************************************\r\n" +
		"*                                                                              *\r\n" +
		"*                              Welcome to SerienHelper                         *\r\n" +
		"*                                                                              *\r\n" +
		"********************************************************************************\r\n" +
		"*                                                                              *\r\n" +
		"* Author: Ten Eniunlsl                                                         *\r\n" +
		"* License: MIT License                                                         *\r\n" +
		"*                                                                              *\r\n" +
		"* SerienHelper is a lightweight console application that helps you keep track  *\r\n" +
		"* of new episodes for your favorite TV series.                                 *\r\n" +
		"*                                                                              *\r\n" +
		"* IMPORTANT: This is NOT a tool for downloading series!                        *\r\n" +
		"* It is designed purely to monitor and inform you about new releases.          *\r\n" +
		"*                                                                              *\r\n" +
		"* SerienHelper provides an easy way to stay updated without the hassle of      *\r\n" +
		"* manually checking for new episodes.                                          *\r\n" +
		"*                                                                              *\r\n" +
		"* Actual implemented: bs.to, s.to and aniworld.to                              *\r\n" +
		"*                                                                              *\r\n" +
		"********************************************************************************";

	static ConsoleColor DefaultColor = Console.ForegroundColor;

	static async Task<int> Main(string[] args)
	{
		var rootCommand = new RootCommand(HeaderText);
		
		RegisterCreateCommand(ref rootCommand);

		RegisterCompareCommand(ref rootCommand);

		RegisterAddCommand(ref rootCommand);

		RegisterShowListCommand(ref rootCommand);

		RegisterVersionCommand(ref rootCommand);

		RegisterGetUpdateInfoCommand(ref rootCommand);

        return await rootCommand.InvokeAsync(args);
	}

	/// <summary>
	/// Register the Version Command, to show the version
	/// </summary>
	/// <param name="rootCommand"></param>
	private static void RegisterVersionCommand(ref RootCommand rootCommand)
	{
		var versionCommand = new Command("version", "Shows the Serienhelper Version");

		versionCommand.SetHandler(() =>
		{
			var versionObj = Assembly.GetExecutingAssembly().GetName().Version;
			var version = versionObj.Major + "." + versionObj.Minor;
			Console.WriteLine($"Version: {version}");
		});
		rootCommand.AddCommand(versionCommand);
	}

	/// <summary>
	/// Registers the Create Command to create series folder with subfolders for seasons and handler.json, series image and description
	/// </summary>
	/// <param name="rootCommand"></param>
	private static void RegisterCreateCommand(ref RootCommand rootCommand)
	{
		var createTypeOption = new Option<string?>(
			name: "type",
			description: "The type to Create");

		var urlOption = new Option<string?>(
			name: "from",
			description: "Creates a new Serie Folder");

		var toOption = new Option<string?>(
			name: "to",
			description: "Parent Folder Path");
		toOption.AddValidator(result => {
			var value = result.GetValueForOption(toOption);

			if (!IsPathValid(value))
			{
				result.ErrorMessage = "The given folder path is invalid!";
			}
		});

		var createCommand = new Command("create", "Creates Serie Folder from Url")
		{
			createTypeOption,
			urlOption,
			toOption
		};
		rootCommand.AddCommand(createCommand);

		createCommand.SetHandler((createType, url, to) =>
		{
			to = CorrectDevicePath(to);
			var handler = Logic.GetHandler(url);

			if (to.EndsWith("\"") && !Directory.Exists(to)) to = to.TrimEnd('"');
			if (createType?.ToLower() == "folder")
			{
				var path = handler.CreateFolderFromUrl(url, to);
				if (!string.IsNullOrWhiteSpace(path))
				{
					Console.WriteLine($"Folder Created at: {path}");
				}
				else
				{
					Console.WriteLine("The folder couldn't be created!");
				}
			}
			else if (createType?.ToLower() == "showinfo")
			{
				handler.CreateShowInfo(url, to);
			}
		},
		createTypeOption, urlOption, toOption);
	}

	/// <summary>
	/// Registers the Comapre Command to compare folders with websites
	/// </summary>
	/// <param name="rootCommand"></param>
	private static void RegisterCompareCommand(ref RootCommand rootCommand)
	{
		var compareTypeOption = new Option<ECompareType>(
			name: "type",
			description: "The compare type like structure or full",
			getDefaultValue: () => { return ECompareType.full; });

		var compareFolderOption = new Option<string>(
			name: "folder",
			description: "The folder to compare");
		compareFolderOption.AddValidator(result => {
			var value = result.GetValueForOption(compareFolderOption);

			if (!IsPathValid(value))
			{
				result.ErrorMessage = "The given folder path is invalid!";
			}
		});

		var infoTypeOption = new Option<EInfoType>(
			name: "info",
			description: "The info type to compare (bs, aniworld)",
			getDefaultValue: () => { return EInfoType.all; });

		var compareAllOption = new Option<bool>(
			name: "all",
			description: "The folder to compare all sub folder")
		{ Arity = ArgumentArity.ZeroOrOne };

		compareAllOption.SetDefaultValue(false);

		var ignoreListOption = new Option<string>(
			name: "ignore",
			description: "a JSON File to ignore on compare");

		var langFilterOption = new Option<ELanguage?>(
			name: "lang",
			description: "filter results by language");

		var outTypeOption = new Option<EOutputType>(
			name: "out",
			description: "The output type: text, html, json");
		outTypeOption.SetDefaultValue(EOutputType.text);

		var replacePathOption = new Option<List<string>>(
			name: "replacepath",
			description: "Replace output paths, oldpath newpath and optional pathdelimer \\ or /")
		{ Arity = new ArgumentArity(2, 3), };
		
		var compareCommand = new Command("compare", "Compares folder with showInfo") {
			compareFolderOption,
			compareTypeOption,
			infoTypeOption,
			compareAllOption,
			ignoreListOption,
			langFilterOption,
			outTypeOption,
			replacePathOption
		};

		replacePathOption.AllowMultipleArgumentsPerToken = true;
				
		compareCommand.SetHandler((folderPath, compareType, infotype, all, ignoreListPath, language, outtype, replacePaths) =>
		{
			folderPath = CorrectDevicePath(folderPath)!;

			if (replacePaths.Count >= 2 && (string.IsNullOrWhiteSpace(replacePaths.FirstOrDefault()) || string.IsNullOrWhiteSpace(replacePaths.LastOrDefault())))
			{
				Console.ForegroundColor = ConsoleColor.Red;
				Console.WriteLine("Arguments on replacepath not set!");
				Console.ForegroundColor = DefaultColor;
				return;
			}

			if (replacePaths.Count == 3 && (replacePaths[2] != "/" && replacePaths[2] != "\\"))
			{
				Console.ForegroundColor = ConsoleColor.Red;
				Console.WriteLine("Path Delimeter musst be / or \\");
				Console.ForegroundColor = DefaultColor;
				return;
			}

			var ignoreList = new IgnoreList();
			if (!string.IsNullOrWhiteSpace(ignoreListPath)) ignoreList = IgnoreList.Load(ignoreListPath);

			var results = new List<CompareResult>();
			if (!all)
			{
				results.Add(Logic.Compare(folderPath, compareType, infotype, language, ignoreList));
			}
			else if (!string.IsNullOrEmpty(folderPath))
			{
				var subFolders = Directory.GetDirectories(folderPath);
				foreach (var subFolder in subFolders)
				{
					if (!ignoreList.Any(x => !x.IsEpisode && x.Path == subFolder))
					{
						results.Add(Logic.Compare(subFolder, compareType, infotype, language, ignoreList));
					}
				}
			}

			var compareOut = GenerateCompareOutput(results, language, folderPath, replacePaths);
			switch (outtype)
			{
				case EOutputType.text:
					PrintCompareText(compareOut);
					break;
				case EOutputType.html:
					PrintCompareHtml(compareOut);
					break;
				case EOutputType.json:
					Console.WriteLine(JsonSerializer.Serialize(compareOut));
					break;
				default:
					PrintCompareText(compareOut);
					break;
			}
		},
		compareFolderOption, compareTypeOption, infoTypeOption, compareAllOption, ignoreListOption, langFilterOption, outTypeOption, replacePathOption);
		rootCommand.AddCommand(compareCommand);
	}

	private static OutputData GenerateCompareOutput(List<CompareResult> results, ELanguage? language, string folderPath, List<string> replacePaths)
	{
		var folderNotFoundList = results.Where(x => !x.IsFolderFound).ToList();
		var showInfoNotFoundList = results.Where(x => !x.IsShowInfoFound).ToList();

		var compareFolder = (replacePaths.Count < 2) ? folderPath : folderPath.Replace(replacePaths[0], replacePaths[1]);
		var folderNotFoundPaths = folderNotFoundList.Select(x => (replacePaths.Count < 2) ? x.FolderPath : x.FolderPath.Replace(replacePaths[0], replacePaths[1])).ToList();
		var showInfoNotFoundPaths = showInfoNotFoundList.Select(x => (replacePaths.Count < 2) ? x.FolderPath : x.FolderPath.Replace(replacePaths[0], replacePaths[1])).ToList();
		if (replacePaths.Count == 3)
		{
			compareFolder = compareFolder.Replace("/", replacePaths[2]).Replace("\\", replacePaths[2]);
			folderNotFoundPaths = folderNotFoundPaths.Select(x => x.Replace("/", replacePaths[2]).Replace("\\", replacePaths[2])).ToList();
			showInfoNotFoundPaths = showInfoNotFoundPaths.Select(x => x.Replace("/", replacePaths[2]).Replace("\\", replacePaths[2])).ToList();
		}

		var output = new OutputData()
		{
			CompareFolder = compareFolder,
			FoldersNotFoundCount = folderNotFoundList.Count,
			FoldersNotFoundPaths = folderNotFoundPaths,
			ShowInfoNotFoundCount = showInfoNotFoundList.Count,
			ShowInfoNotFoundPaths = showInfoNotFoundPaths,
		};

		results.Where(x => x.StructureIssues.Count > 0 || x.SeasonStructureIssues.Where(y => language == null || y.Value == language).Count() > 0).ToList().ForEach((x) =>
		{
			var path = x.FolderPath;
			var structureIssues = x.StructureIssues;
			var seasonStructureIssues = ((language == null)
					? x.SeasonStructureIssues.Keys.ToList()
					: x.SeasonStructureIssues.Where(z => z.Value == language).Select(z => z.Key).ToList());

			if (replacePaths.Count >= 2) 
			{
				path = path.Replace(replacePaths[0], replacePaths[1]);
				structureIssues = x.StructureIssues.Select(y => y.Replace(replacePaths[0], replacePaths[1])).ToList();
				seasonStructureIssues = seasonStructureIssues.Select(y => y.Replace(replacePaths[0], replacePaths[1])).ToList();
			}

			if(replacePaths.Count == 3)
			{
				path = path.Replace("/", replacePaths[2]).Replace("\\", replacePaths[2]);
				structureIssues = structureIssues.Select(y => y.Replace("/", replacePaths[2]).Replace("\\", replacePaths[2])).ToList();
				seasonStructureIssues = seasonStructureIssues.Select(y => y.Replace("/", replacePaths[2]).Replace("\\", replacePaths[2])).ToList();
			}

			var data = new OutputStructureIssue
			{
				Path = path,
				StructureIssues = structureIssues,
				SeasonStructureIssues = seasonStructureIssues,
			};
			output.StructureIssues.Add(data);
		});

		results.Where(x => x.EpisodesNotFound.Count > 0).ToList().ForEach((x) =>
		{
			foreach (var epi in x.EpisodesNotFound)
			{
				var issues = (language == null) ? epi.Value : epi.Value.Where(z => z.Languages.Contains((ELanguage)language)).ToList();
				if (issues.Count == 0) break;

				var path = (replacePaths.Count < 2) ? x.FolderPath : x.FolderPath.Replace(replacePaths[0], replacePaths[1]);
				if (replacePaths.Count == 3) path = path.Replace("/", replacePaths[2]).Replace("\\", replacePaths[2]);

				var episodesBlock = new OutputEpisodesBlock()
				{
					Count = issues.Count,
					Title = epi.Key.Title,
					Path = path,
				};
				output.EpisodesBlocks.Add(episodesBlock);

				var outSeason = new OutputSeason();
				foreach (var issue in issues)
				{
					if (issue.Season != null && outSeason.Nr != issue.Season.Nr)
					{
						outSeason.Nr = issue.Season.Nr;
						outSeason.Title = issue.Season.Title;
						
						episodesBlock.Seasons.Add(outSeason);
					}
					string title = (!string.IsNullOrWhiteSpace(issue.Title)) ? issue.Title : issue.AlternateTitle;

					outSeason.Episodes.Add(new OutputEpisode()
					{
						Nr = issue.Nr,
						Title = title,
						Url = issue.Url,
					});
				}
			}
		});

		return output;
	}

	private static void PrintCompareHtml(OutputData compareOutput)
	{
		var template = GetTemplate("HTML.Template.html");
		var structureIssueTemplate = GetTemplate("HTML.StructureIssueTemplate.html");
		var episodesTemplate = GetTemplate("HTML.EpisodesTemplate.html");

		string structureIssues = "";

		compareOutput.StructureIssues.ForEach(x => {
			var data = new
			{
				path = x.Path,
				structureIssues = string.Join("\n", x.StructureIssues.Select(z => $"<li>{z}</li>")),
				seasonStructureIssues = string.Join("\n", x.SeasonStructureIssues.Select(z => $"<li>{z}</li>")),
			};
			structureIssues += ReplacePlaceHolders(structureIssueTemplate, data) + "\n";
		});

		string episodesNotFound = "";
		compareOutput.EpisodesBlocks.ForEach(x =>
		{
			string episodeBlock = "";
			x.Seasons.ForEach(season => 
			{
				string seasonHead = $"<tr><th>Season</th><th>Title</th><th>{((!string.IsNullOrWhiteSpace(season.Path)) ? "Path": "")}</th>";
				string seasonPathPart = (!string.IsNullOrWhiteSpace(season.Path)) ? $"<a href=\"file://{season.Path}\">{season.Path}</a>" : "";
				string seasonRow = $"<tr><td>{season.Nr}</td><td>{season.Title}</td><td></td></tr>\n";
				string episodeRows = $"<tr><th>Nr</th><th>Title</th><th>Url</th></tr>\n";

				season.Episodes.ForEach(episode =>
				{
					episodeRows += $"<tr><td>{episode.Nr}</td><td>{episode.Title}</td><td><a href=\"{episode.Url}\">{episode.Url}</a></td></tr>\n";
				});
				episodeBlock += (seasonHead + seasonRow + episodeRows);
			});

			var data = new
			{
				EpisodesNotFoundCount = x.Count.ToString(),
				EpisodesNotFoundSerieTitle = x.Title,
				SeriesPath = $"<a href=\"file://{x.Path}\">" + x.Path + "</a>",
				EpisodeBlock = episodeBlock 
			};

			episodesNotFound += ReplacePlaceHolders(episodesTemplate, data);
		});

		var data = new
		{
			Date = compareOutput.Date.ToString("yyyy-MM-ddTHH:mm:ssK"),
			Version = compareOutput.Version,
			CompareFolder = compareOutput.CompareFolder,
			FoldersNotFoundCount = compareOutput.FoldersNotFoundCount,
			FoldersNotFoundPaths = string.Join("", compareOutput.FoldersNotFoundPaths.Select(x => $"<li>{x}</li>")),
			ShowInfoNotFoundCount = compareOutput.ShowInfoNotFoundCount,
			ShowInfoNotFoundPaths = string.Join("", compareOutput.ShowInfoNotFoundPaths.Select(x => $"<li><a href=\"file://{x}\">{x}</a></li>")),
			StructureIssues = structureIssues,
			EpisodesNotFound = episodesNotFound,
		};

		Console.WriteLine(ReplacePlaceHolders(template, data));
	}

	private static void PrintCompareText(OutputData outputData)
	{
		outputData.FoldersNotFoundPaths.ForEach(x => Console.WriteLine($"Folder does'nt exist! {x}"));
		Console.WriteLine();

		outputData.ShowInfoNotFoundPaths.ForEach(x => Console.WriteLine($"Show info not found! {x}"));
		Console.WriteLine();
		
		outputData.StructureIssues.ForEach((x) => { 
			Console.WriteLine($"Structure Issues ({x.Path}):");
			
			x.SeasonStructureIssues.ForEach((y) => Console.WriteLine(y));
			x.StructureIssues.ForEach(y => Console.WriteLine(y));
			Console.WriteLine();
		});
		
		outputData.EpisodesBlocks.ForEach((x) => {
			Console.ForegroundColor = ConsoleColor.Green;
			Console.WriteLine($"Episodes Not found ({x.Count}): {x.Title}");
			Console.ForegroundColor = DefaultColor;

			x.Seasons.ForEach((season) => {
				Console.WriteLine("");
				Console.WriteLine($"Season: {season.Nr} Title: {season.Title}");

				season.Episodes.ForEach((episode) =>
				{
					Console.WriteLine($"Nr {episode.Nr}, Title {episode.Title}, Url {episode.Url}");
				});
			});
			Console.WriteLine();
		});
	}

	/// <summary>
	/// Registers the Add Command to add Series or Episodes to ignorelist
	/// </summary>
	/// <param name="rootCommand"></param>
	private static void RegisterAddCommand(ref RootCommand rootCommand)
	{
		var addCommand = new Command("add", "adds an entry to ignorelist");
		var folderOption = new Option<string>(
			name: "folder",
			description: "folder to add");
		folderOption.IsRequired = true;
		folderOption.AddValidator(result => {
			var value = result.GetValueForOption(folderOption);

			if (!IsPathValid(value))
			{
				result.ErrorMessage = "The given folder path is invalid!";
			}
		});
		addCommand.AddOption(folderOption);

		var episodeOption = new Option<int?>(
			name: "episode",
			description: "episode to add");
		addCommand.AddOption(episodeOption);

		var ignoreListOption = new Option<string>(
			name: "ignore",
			description: "a JSON File to ignore on compare",
			getDefaultValue: () => { return "ignore.json"; });
		addCommand.AddOption(ignoreListOption);

		addCommand.SetHandler((folder, episodeNr, ignoreListPath) =>
		{
			folder = CorrectDevicePath(folder)!;

			var ignoreList = new IgnoreList();
			if (!string.IsNullOrWhiteSpace(ignoreListPath) && File.Exists(ignoreListPath)) ignoreList = IgnoreList.Load(ignoreListPath);

			if (episodeNr != null)
			{
				if (!ignoreList.Any(x => x.Path == folder && x.EpisodeNr == episodeNr))
				{
					ignoreList.Add(new(folder) { EpisodeNr = episodeNr });
					ignoreList.Save(ignoreListPath);
				}
			}
			else if (!string.IsNullOrWhiteSpace(folder))
			{
				if (folder.EndsWith("\"") && !Directory.Exists(folder)) folder = folder.TrimEnd('"');

				if (!ignoreList.Any(x => x.Path == folder))
				{
					ignoreList.Add(new(folder));
					ignoreList.Save(ignoreListPath);
				}
			}
			else
			{
				Console.WriteLine("File or Folder was an empty string!");
			}
		},
		folderOption, episodeOption, ignoreListOption);
		rootCommand.AddCommand(addCommand);
	}

	/// <summary>
	/// Regsiters the ShowListCommand to filter get a filtered List of Series by Genres
	/// </summary>
	/// <param name="rootCommand"></param>
	/// <exception cref="NotImplementedException">If handler not supports filtering</exception>
	private static void RegisterShowListCommand(ref RootCommand rootCommand)
	{
        var handlerOption = new Option<EInfoType>(
            name: "handler",
            description: "The Handler");

        var genresOption = new Option<string>(
            name: "genres",
            description: "possible values a comma seperated list action,ger");
		genresOption.IsRequired = true;

        var folderOption = new Option<string?>(
            name: "folder",
            description: "If folder option is set, only not existing Series will be displayed");
		folderOption.AddValidator(result => {
			var value = result.GetValueForOption(folderOption);

			if (!IsPathValid(value))
			{
				result.ErrorMessage = "The given folder path is invalid!";
			}
		});
		var showListCommand = new Command("showlist", "Displays a List of Series or Animes by filtering") {
            handlerOption,
			genresOption,
			folderOption,
        };

		showListCommand.SetHandler((handlertype, filter, folder) =>
		{
            var handler = Logic.GetHandler(handlertype) as IFilterableListingHandler;
			if(handler == null)
			{
				throw new NotImplementedException($"Filtering is not implemented for {handlertype}");
			}

			var genres = filter.Split(",");
			folder = CorrectDevicePath(folder);
			List<ShowInfo> showInfos = new List<ShowInfo>();
			if (!string.IsNullOrWhiteSpace(folder)) showInfos = Logic.GetAllLocalShowInfos(folder, handlertype);

			var result = new Dictionary<string, string>();
			var seriesDict = new Dictionary<string, Dictionary<string, string>>();

			foreach(var genre in genres)
			{
				seriesDict.Add(genre, handler.GetSeriesByGenre(genre));
			}

			var seriesByFirstGenre = seriesDict.First();
			foreach (var serie in seriesByFirstGenre.Value)
			{
				bool hasFound = true;
				foreach (var series in seriesDict.Where(x => x.Key != seriesByFirstGenre.Key))
				{
					if (!series.Value.ContainsKey(serie.Key))
					{
						hasFound = false;
						break;
					}
				}

				if (hasFound && !showInfos.Any(x => x.Url.Contains(serie.Key))) result.Add(serie.Key, serie.Value);
			}

			Console.ForegroundColor = ConsoleColor.Green;
			Console.WriteLine($"Series found by filter: {result.Count}, Series in Folder: {showInfos.Count}");
			Console.ForegroundColor = DefaultColor;

			foreach(var serie in result)
			{
				Console.WriteLine(serie.Value + ": " + serie.Key);
			}

		}, handlerOption, genresOption, folderOption);

		rootCommand.AddCommand(showListCommand);
    }

	private static void RegisterGetUpdateInfoCommand(ref RootCommand rootCommand)
	{
		var handlerOption = new Option<EInfoType>(
		   name: "handler",
		   description: "The Handler");

		var allOption = new Option<bool>(
			name: "all",
			description: "to get all sub folder")
		{ Arity = ArgumentArity.ZeroOrOne };

		allOption.SetDefaultValue(false);

		var folderOption = new Option<string>(
			name: "folder",
			description: "The folder");
		folderOption.AddValidator(result => {
			var value = result.GetValueForOption(folderOption);

			if (!IsPathValid(value))
			{
				result.ErrorMessage = "The given folder path is invalid!";
			}
		});

		var compareCommand = new Command("getupdateinfo", "Gets Path and URLs as JSON to update folder after Version change") {
			handlerOption,
			allOption,
			folderOption
		};

		compareCommand.SetHandler((handler, all, folder) =>
		{
			var dict = new Dictionary<string, List<string>>();
			if (all)
			{
				foreach (var subfolder in Directory.GetDirectories(folder))
				{
					var localShowInfos = Logic.GetLocalShowInfo(subfolder, handler);
					dict[subfolder] = localShowInfos.Select(x => x.Url).ToList();
				}
			}
			else
			{
				var localShowInfos = Logic.GetLocalShowInfo(folder, handler);
				dict[folder] = localShowInfos.Select(x => x.Url).ToList();
			}

			Console.WriteLine(JsonSerializer.Serialize(dict));

		}, handlerOption, allOption, folderOption);
		rootCommand.AddCommand(compareCommand);
	}

	private static string ReplacePlaceHolders(string template, object data)
	{
		string result = template;
		var propperties = data.GetType().GetProperties();
		foreach (var prop in propperties)
		{
			string? val = prop.GetValue(data, null)?.ToString();
			result = result.Replace("{" + prop.Name + "}", (!string.IsNullOrWhiteSpace(val) ? val : ""));
		}
		return result;
	}

	public static bool IsPathValid(string? path)
	{
		if (string.IsNullOrWhiteSpace(path))
		{
			return false;
		}
		else
		{
			var inavlidPathChars = Path.GetInvalidPathChars().ToList();
			inavlidPathChars.Add('\"');
			foreach (var c in inavlidPathChars)
			{
				if (path.Contains(c)) return false;
			}
		}
		return true;
	}

	public static string? CorrectDevicePath(string? path)
	{
		if (string.IsNullOrWhiteSpace(path)) return path;
		if (path.Length > 2) return path;

		var rgx = new Regex(@"[a-zA-Z]:");
		if (!rgx.IsMatch(path)) return path;

		return path.Replace(":", ":\\");
	}

	public static string GetTemplate(string name)
	{
		var assembly = Assembly.GetExecutingAssembly();
		string resourcename = $"{assembly.GetName().Name}.{name}";
		using var stream = assembly.GetManifestResourceStream(resourcename)!;

		using var reader = new StreamReader(stream, Encoding.UTF8);
		return reader.ReadToEnd();
	}
}