﻿using SerienHelper.API.Enum;
using SerienHelper.API.Handling;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using static SerienHelper.API.Handling.BaseWebsiteHandler;

namespace SerienHelper.API
{
	public static class Logic
	{
		public static List<ShowInfo> GetLocalShowInfo(string folderPath, EInfoType infoType)
		{
			var localShowInfos = new List<ShowInfo>();
			var infoFiles = Directory.GetFiles(folderPath, "info*.json").ToList();

			if (infoType != EInfoType.all)
			{
				infoFiles.RemoveAll(x => !Path.GetFileName(x).Contains(infoType.ToString()));
			}

			foreach (var infoFile in infoFiles)
			{
				localShowInfos.Add(ShowInfo.Load(infoFile));
			}
			return localShowInfos;
		}

		public static List<ShowInfo> GetAllLocalShowInfos(string folderPath, EInfoType infoType)
		{
            var localShowInfos = new List<ShowInfo>();
            foreach (var directory in Directory.GetDirectories(folderPath))
			{
				var showInfos = GetLocalShowInfo(directory, infoType);
				localShowInfos.AddRange(showInfos);
			}
			return localShowInfos;
		}

		public static CompareResult Compare(string folderPath, ECompareType compareType, EInfoType infoType, ELanguage? langFilter, IgnoreList ignoreList)
		{
			var compareResult = new CompareResult();
			compareResult.FolderPath = folderPath;

			if (!Directory.Exists(folderPath)) return compareResult;

			compareResult.IsFolderFound = true;

			var localShowInfos = GetLocalShowInfo(folderPath, infoType);
			if (localShowInfos.Count == 0)
			{
				return compareResult;
			}
			compareResult.IsShowInfoFound = true;


			foreach (var localShowInfo in localShowInfos)
			{
				var url = localShowInfo.Url;

				var externalShowInfo = Logic.GetHandler(url).GetShowInfo(url);

				var comparedSeasonFolders = CompareSeasonFolderStructure(externalShowInfo, folderPath);
				foreach (var compared in comparedSeasonFolders)
				{
					if (!compareResult.SeasonStructureIssues.ContainsKey(compared.Key)) compareResult.SeasonStructureIssues.Add(compared.Key, compared.Value);
				}
				compareResult.StructureIssues.AddRange(CompareStructure(externalShowInfo!, folderPath));

				var episodesNotFound = new List<Episode>();

				if (compareType == ECompareType.full)
				{
					foreach (var season in externalShowInfo!.Seasons)
					{
						if (season.SeasonType == Season.ESeasonType.Season)
						{
							if (season.Languages.Contains(ELanguage.Deutsch))
							{
								string? seasonPath = GetGermanSeasonDirectory(season, folderPath);
								if (!string.IsNullOrWhiteSpace(seasonPath) && Directory.Exists(seasonPath))
								{
									var episodeFiles = Directory.GetFiles(seasonPath, "S*.mp4");
									foreach (var episode in season.Episodes.Where(x => x.Languages.Contains(ELanguage.Deutsch)))
									{
										string episodeNr = "S" + season.Nr.ToString("D2") + "E" + episode.Nr.ToString("D2");
										if (!episodeFiles.Any(x => Path.GetFileName(x).StartsWith(episodeNr)))
										{
											if (!ignoreList.Any(x => x.EpisodeNr == episode.Nr && x.Path.Contains(seasonPath))) episodesNotFound.Add(episode);
										}
									}
								}
							}

							var nonGermanSeason = GetNonGermanSeasonDirectory(season, folderPath);
							if (nonGermanSeason != null && !string.IsNullOrWhiteSpace(nonGermanSeason.Value.Key) && Directory.Exists(nonGermanSeason.Value.Key))
							{
								var episodeFiles = Directory.GetFiles(nonGermanSeason.Value.Key, "S*.mp4");
								foreach (var episode in season.Episodes.Where(x => !x.Languages.Contains(ELanguage.Deutsch)))
								{
									string episodeNr = "S" + season.Nr.ToString("D2") + "E" + episode.Nr.ToString("D2");
									if (!episodeFiles.Any(x => Path.GetFileName(x).StartsWith(episodeNr)))
									{
										if (!ignoreList.Any(x => x.EpisodeNr == episode.Nr && x.Path.Contains(nonGermanSeason.Value.Key))) episodesNotFound.Add(episode);
									}
								}
							}
						}
						else if (season.SeasonType == Season.ESeasonType.Special)
						{
							if (season.Languages.Contains(ELanguage.Deutsch))
							{
								string? seasonPath = GetGermanSeasonDirectory(season, folderPath);
								if (!string.IsNullOrWhiteSpace(seasonPath) && Directory.Exists(seasonPath))
								{
									var episodeFiles = Directory.GetFiles(seasonPath, "*.mp4");
									foreach (var episode in season.Episodes.Where(x => x.Languages.Contains(ELanguage.Deutsch)))
									{
										string episodeNr = "E" + episode.Nr.ToString("D2");
										if (!episodeFiles.Any(x => Path.GetFileName(x).StartsWith(episodeNr)))
										{
											if (!ignoreList.Any(x => x.EpisodeNr == episode.Nr && x.Path.Contains(seasonPath))) episodesNotFound.Add(episode);
										}
									}
								}
							}

							var nonGermanSeason = GetNonGermanSeasonDirectory(season, folderPath);
							if (nonGermanSeason != null && !string.IsNullOrWhiteSpace(nonGermanSeason.Value.Key) && Directory.Exists(nonGermanSeason.Value.Key))
							{
								var episodeFiles = Directory.GetFiles(nonGermanSeason.Value.Key, "*.mp4");
								foreach (var episode in season.Episodes.Where(x => !x.Languages.Contains(ELanguage.Deutsch)))
								{
									string episodeNr = "E" + episode.Nr.ToString("D2");
									if (!episodeFiles.Any(x => Path.GetFileName(x).StartsWith(episodeNr)))
									{
										if(!ignoreList.Any(x => x.EpisodeNr == episode.Nr && x.Path.Contains(nonGermanSeason.Value.Key))) episodesNotFound.Add(episode);
									}
								}
							}
						}
					}

					if (episodesNotFound.Count > 0)
					{
						compareResult.EpisodesNotFound.Add(localShowInfo, episodesNotFound);
					}
				}
			}


			return compareResult;
		}

		private static Dictionary<string, ELanguage> CompareSeasonFolderStructure(ShowInfo showInfo, string folderPath)
		{
			var retVal = new Dictionary<string, ELanguage>();
			foreach (var season in showInfo.Seasons)
			{
				if (season.Languages.Contains(ELanguage.Deutsch))
				{
					string? seasonPath = GetGermanSeasonDirectory(season, folderPath);
					if (!string.IsNullOrWhiteSpace(seasonPath) && !Directory.Exists(seasonPath)) retVal.Add(seasonPath, ELanguage.Deutsch);
				}

				var nonGermanSeason = GetNonGermanSeasonDirectory(season, folderPath);
				if (nonGermanSeason != null && !Directory.Exists(((KeyValuePair<string, ELanguage>)nonGermanSeason).Key))
				{
					retVal.Add(nonGermanSeason.Value.Key, nonGermanSeason.Value.Value);
				}
			}
			return retVal;
		}

		private static List<string> CompareStructure(ShowInfo showInfo, string folderPath)
		{
			var retVal = new List<string>();

			if (!string.IsNullOrWhiteSpace(showInfo.Description))
			{
				var textPath = Path.Combine(folderPath, "info.txt");
				if (!File.Exists(textPath)) retVal.Add(textPath);
			}

			if (!string.IsNullOrWhiteSpace(showInfo.ImageUrl))
			{
				var jpgPath = Path.Combine(folderPath, "folder.jpg");
				if (!File.Exists(jpgPath)) retVal.Add(jpgPath);
			}

			return retVal;
		}


		public static string? GetGermanSeasonDirectory(Season season, string folderPath)
		{
			if (season.SeasonType == Season.ESeasonType.Season)
			{
				var seasonPath = Path.Combine(folderPath, season.Nr.ToString("D2"));
				return seasonPath;
			}
			else if (season.SeasonType == Season.ESeasonType.Special)
			{
				var specialPath = Path.Combine(folderPath, "special");
				return specialPath;
			}
			return null;
		}

		public static KeyValuePair<string, ELanguage>? GetNonGermanSeasonDirectory(Season season, string folderPath)
		{
			var germanEpisodes = season.Episodes.Where(x => x.Languages.Contains(ELanguage.Deutsch)).ToList();
			var nonGermanEpisodes = season.Episodes.Where(x => !x.Languages.Contains(ELanguage.Deutsch)).ToList();
			if (nonGermanEpisodes.Count > 0)
			{
				if (season.SeasonType == Season.ESeasonType.Season)
				{
					return new(Path.Combine(folderPath, season.Nr.ToString("D2") + " gersub"), ELanguage.Japanisch_Deutsch);
				}
				else if (season.SeasonType == Season.ESeasonType.Special)
				{
					return new(Path.Combine(folderPath, "special gersub"), ELanguage.Japanisch_Deutsch);
				}
			}
			return null;
		}

		public static API.Handling.BaseWebsiteHandler GetHandler(string url)
		{
			if (url.Contains("bs.to"))
			{
				return new BSHandler();
			}
			else if (url.Contains("aniworld.to"))
			{
				return new AniworldHandler();
			}
			else if (url.Contains("s.to"))
			{
				return new STOHandler();
			}
			return null;
		}

		public static API.Handling.BaseWebsiteHandler GetHandler(EInfoType infoType)
		{
			if (infoType == EInfoType.bs)
			{
				return new BSHandler();
			}
			else if (infoType == EInfoType.aniworld)
			{
				return new AniworldHandler();
			}
			else if (infoType == EInfoType.sto)
			{
				return new STOHandler();
			}
			return null;
		}
	}
}
