﻿using HtmlAgilityPack;
using SerienHelper.API.Interfaces;
using System.Net;
using System.Text.RegularExpressions;

namespace SerienHelper.API.Handling
{
	public class STOHandler : BaseWebsiteHandler, IHandler, IFilterableListingHandler
    {
        private string _BaseUrl = "https://s.to";

        public override ShowInfo GetShowInfo(string url, bool getEpisodeDesc = true)
        {
            var client = new WebClient();
            client.Headers.Add("Accept-Language", "de-de,de;q=0.8,en-us;q=0.5,en;q=0.3");
            var html = client.DownloadString(url);

            return GetShowInfoFromHtml(html, url, getEpisodeDesc);
        }

        public override string GetShowInfoFilename()
        {
            return "info_sto.json";
        }

        public override ShowInfo GetShowInfoFromHtml(string html, string url, bool getEpisodeDesc = true)
        {
            var titleMatch = Regex.Match(html, "class=\"series-title\".*?<span>(.*?)</span>", RegexOptions.Multiline);
            var descriptionMatch = Regex.Match(html, "data-full-description=\"(.*?)\">");
            var imageUrlMatch = Regex.Match(html, "class=\"seriesCoverBox\".*?<img.*?data-src=\"(.*?)\"");
            var productionStartMatch = Regex.Match(html, @"itemprop=""startDate"">.*?<a.*?>.*?(\d\d\d\d).*?</a>");
            var productionEndMatch = Regex.Match(html, "itemprop=\"endDate\">.*?<a.*?>.*?(\\d\\d\\d\\d).*?</a>");
            var regieMatch = Regex.Matches(html, "itemprop=\"director\".*?itemprop=\"name\">(.*?)</span>");
            var artistsMatch = Regex.Matches(html, "itemprop=\"actor\".*?itemprop=\"name\">(.*?)</span>");
            var producerMatch = Regex.Matches(html, "itemprop=\"creator\".*?itemprop=\"name\">(.*?)</span>");
            var genreMatch = Regex.Matches(html, "itemprop=\"genre\">(.*?)</a>");

            var showInfo = new ShowInfo()
            {
                Title = titleMatch.Groups[1].Value,
                Description = descriptionMatch.Groups[1].Value,
                Url = url,
                ImageUrl = _BaseUrl + imageUrlMatch.Groups[1].Value,
                ProductionStart = productionStartMatch.Groups[1].Value,
                ProductionEnd = productionEndMatch.Groups[1].Value,
                Regie = regieMatch.Select(x => x.Groups[1].Value).ToList(),
                Artists = artistsMatch.Select(x => x.Groups[1].Value).ToList(),
                Producer = producerMatch.Select(x => x.Groups[1].Value).ToList(),
                Genre = genreMatch.Select(x => x.Groups[1].Value).ToList(),
            };

            var matchSeasonsBlock = Regex.Match(html, @"Staffeln:(.*?)</ul>", RegexOptions.Singleline);
            var matchSeasons = Regex.Matches(matchSeasonsBlock.Groups[1].Value, @"href=""(.*?)"".*?title=""(.*?)"".*?>(.*?)</a>");

            int seasonNr;
            showInfo.Seasons = matchSeasons.Select(x => new Season()
            {
                Url = _BaseUrl + x.Groups[1].Value,
                Title = x.Groups[2].Value,
                Nr = int.TryParse(x.Groups[3].Value, out seasonNr) ? seasonNr : 0,
                SeasonType = int.TryParse(x.Groups[3].Value, out seasonNr) ? Season.ESeasonType.Season : Season.ESeasonType.Special
            }).ToList();

            foreach (var season in showInfo.Seasons)
            {
                var client = new WebClient();
                var seasonHtml = client.DownloadString(season.Url);

                var episodesBlockMatch = Regex.Match(seasonHtml, @"class=""seasonEpisodesList""(.*?)</table>", RegexOptions.Singleline);
                var episodeBlockMatch = Regex.Matches(episodesBlockMatch.Groups[1].Value, @"itemprop=""episode""(.*?)</tr>", RegexOptions.Singleline);

                for (int i = 0; i < episodeBlockMatch.Count; i++)
                {
                    Match episodeBlock = episodeBlockMatch[i];
                    var episodesTitleMatch = Regex.Match(episodeBlock.Groups[1].Value, @"""seasonEpisodeTitle"".*?<strong>(.*?)</strong>.*?<span>(.*?)</span>", RegexOptions.Singleline);
                    var matchLanguages = Regex.Matches(episodeBlock.Groups[1].Value, @"class=""flag"".*?src="".*?/(.*?)\.svg", RegexOptions.Singleline);
                    var matchEpisodeUrl = Regex.Match(episodeBlock.Groups[1].Value, @"<a.*?href=""(.*?)"".*?>", RegexOptions.Singleline);

                    var languageList = new List<ELanguage>();
                    foreach (Match languageMatch in matchLanguages)
                    {
                        string value = languageMatch.Groups[1].Value.ToLower();
						if (value.Contains("japanese-german"))
						{
							languageList.Add(ELanguage.Japanisch_Deutsch);
						}
						else if (value.Contains("german"))
						{
							languageList.Add(ELanguage.Deutsch);
						}
                        else if (value.Contains("japanese-english"))
                        {
                            languageList.Add(ELanguage.Japanisch_Englisch);
                        }
                        else if (value.Contains("english"))
                        {
                            languageList.Add(ELanguage.Englisch);
                        }
                    }

                    string episodeDesc = "";
                    if (getEpisodeDesc)
                    {
                        var episodeUrl = _BaseUrl + matchEpisodeUrl.Groups[1].Value;
                        var episodeHtml = client.DownloadString(episodeUrl);

                        var matchEpisodeDesc = Regex.Match(episodeHtml, @"itemprop=""description"".*?>(.*?)</p>");
                        episodeDesc = matchEpisodeDesc.Groups[1].Value;
                    }

                    season.Episodes.Add(new Episode()
                    {
                        Nr = i + 1,
                        Title = episodesTitleMatch.Groups[1].Value,
                        AlternateTitle = episodesTitleMatch.Groups[1].Value,
                        Url = _BaseUrl + matchEpisodeUrl.Groups[1].Value,
                        Languages = languageList,
                        Description = episodeDesc,
                        Season = season,
                    });
                }
            }

            return showInfo;
        }

        public Dictionary<string, string> GetSeriesByGenre(string genre)
        {
            var result = new Dictionary<string, string>();
            var client = new WebClient();
            client.Headers.Add("Accept-Language", "de-de,de;q=0.8,en-us;q=0.5,en;q=0.3");
            var url = _BaseUrl + "/genre/" + genre;

            for (int i = 1; i < 1000; i++)
            {
                try
                {
                    var html = client.DownloadString(url + "/" + i);
                    var doc = new HtmlDocument();
                    doc.LoadHtml(html);

                    var seriesListContainer = doc.DocumentNode.Descendants().FirstOrDefault(x => x.HasClass("seriesListContainer"));
                    if (seriesListContainer != null && seriesListContainer.HasChildNodes)
                    {
                        foreach (var serieNode in seriesListContainer.ChildNodes)
                        {
                            var linkNode = serieNode.ChildNodes.FirstOrDefault(x => x.Name == "a");
                            var h3Node = serieNode.ChildNodes.FirstOrDefault(x => x.Name == "h3");

                            if (linkNode != null && h3Node != null)
                            {
                                result.Add(linkNode.GetAttributeValue("href", ""), h3Node.InnerText);
                            }
                        }
                    }
                }
                catch (Exception ex)
                {
                    break;
                }
            }
            return result;
        }
    }
}
