﻿using SerienHelper.API.Enum;
using SerienHelper.API.Interfaces;
using System.Net;
using System.Text.Json;

namespace SerienHelper.API.Handling
{
	public abstract class BaseWebsiteHandler : IHandler
    {
        public string? CreateFolderFromUrl(string url, string to = "")
        {
            ShowInfo? showInfo = GetShowInfo(url);

            if (showInfo == null)
            {
                Console.WriteLine("Keine ShowInfo ;-(");
                return null;
            }

            string? folderPath = CreateFolderStructure(showInfo, to);

            if(string.IsNullOrWhiteSpace(folderPath)) return null;

            SaveShowInfo(showInfo, folderPath);

            if (!string.IsNullOrWhiteSpace(showInfo.Description))
            {
                File.WriteAllText(Path.Combine(folderPath, "info.txt"), showInfo.Description);
            }

            if (!string.IsNullOrWhiteSpace(showInfo.ImageUrl)) DownloadImage(showInfo, folderPath);

            return folderPath;
        }

        private void SaveShowInfo(ShowInfo showInfo, string to)
        {
            var json = JsonSerializer.Serialize(showInfo);

            string filename = GetShowInfoFilename();

            if (string.IsNullOrWhiteSpace(filename))
            {
                Console.WriteLine("No Showinfo filename!");
                return;
            }

            File.WriteAllText(Path.Combine(to, filename), json);
        }

        public void CreateShowInfo(string url, string to = "")
        {
            if (string.IsNullOrWhiteSpace(to)) to = "";
            ShowInfo? showInfo = GetShowInfo(url);

            if (showInfo == null)
            {
                Console.WriteLine("Keine ShowInfo ;-(");
                return;
            }

            SaveShowInfo(showInfo, to);
            Console.WriteLine("ShowInfo created");
        }

        private string? CreateFolderStructure(ShowInfo showInfo, string to = "")
        {
            if (to == null) to = "";

            if (string.IsNullOrWhiteSpace(showInfo.Title)) return null;

            var folderTitle = showInfo.Title.Replace(":", "");
            var invalidChars = Path.GetInvalidPathChars().ToList();
            invalidChars.AddRange(Path.GetInvalidFileNameChars());
            foreach (var c in invalidChars)
            {
                folderTitle = folderTitle.Replace(c.ToString(), "");
            }

            var folderPath = Path.Combine(to, folderTitle);
            foreach (var sign in Path.GetInvalidPathChars())
            {
                folderPath = folderPath.Replace(sign.ToString(), "");
            }

            Directory.CreateDirectory(folderPath);
            foreach (var season in showInfo.Seasons)
            {
                if (season.Languages.Contains(ELanguage.Deutsch))
                {
                    string? seasonPath = GetGermanSeasonDirectory(season, folderPath);
                    if (!string.IsNullOrWhiteSpace(seasonPath) && !Directory.Exists(seasonPath)) Directory.CreateDirectory(seasonPath);
                }

                var nonGermanSeason = GetNonGermanSeasonDirectory(season, folderPath);
                if (nonGermanSeason != null && !Directory.Exists(((KeyValuePair<string, ELanguage>)nonGermanSeason).Key))
                {
                    Directory.CreateDirectory(((KeyValuePair<string, ELanguage>)nonGermanSeason).Key);
                }
            }
            return folderPath;
        }

        private string? GetGermanSeasonDirectory(Season season, string folderPath)
        {
            if (season.SeasonType == Season.ESeasonType.Season)
            {
                var seasonPath = Path.Combine(folderPath, season.Nr.ToString("D2"));
                return seasonPath;
            }
            else if (season.SeasonType == Season.ESeasonType.Special)
            {
                var specialPath = Path.Combine(folderPath, "special");
                return specialPath;
            }
            return null;
        }

        private KeyValuePair<string, ELanguage>? GetNonGermanSeasonDirectory(Season season, string folderPath)
        {
            var germanEpisodes = season.Episodes.Where(x => x.Languages.Contains(ELanguage.Deutsch)).ToList();
            var nonGermanEpisodes = season.Episodes.Where(x => !x.Languages.Contains(ELanguage.Deutsch)).ToList();
            if (nonGermanEpisodes.Count > 0)
            {
                if (season.SeasonType == Season.ESeasonType.Season)
                {
                    return new(Path.Combine(folderPath, season.Nr.ToString("D2") + " gersub"), ELanguage.Japanisch_Deutsch);
                }
                else if (season.SeasonType == Season.ESeasonType.Special)
                {
                    return new(Path.Combine(folderPath, "special gersub"), ELanguage.Japanisch_Deutsch);
                }
            }
            return null;
        }

        private void DownloadImage(ShowInfo showInfo, string folderpath)
        {
            var client = new WebClient();
            client.DownloadFile(showInfo.ImageUrl, Path.Combine(folderpath, "folder.jpg"));
        }

        public List<ShowInfo> GetLocalShowInfos(string folderPath, EInfoType infoType)
        {
            var localShowInfos = new List<ShowInfo>();
            var infoFiles = Directory.GetFiles(folderPath, "info*.json").ToList();

            if (infoType != EInfoType.all)
            {
                infoFiles.RemoveAll(x => !Path.GetFileName(x).Contains(infoType.ToString()));
            }

            foreach (var infoFile in infoFiles)
            {
                localShowInfos.Add(ShowInfo.Load(infoFile));
            }
            return localShowInfos;
        }

        public virtual ShowInfo GetShowInfo(string url, bool getEpisodeDesc = true)
        {
            throw new NotImplementedException();
        }

        public virtual ShowInfo GetShowInfoFromHtml(string html, string url, bool getEpisodeDesc = true)
        {
            throw new NotImplementedException();
        }

        public virtual string GetShowInfoFilename()
        {
            throw new NotImplementedException();
        }

        public class CompareResult
        {
            public bool IsFolderFound { get; set; }
            public string FolderPath { get; set; } = string.Empty;
            public bool IsShowInfoFound { get; set; }
            public ShowInfo? ShowInfo { get; set; }

            public List<string> StructureIssues { get; set; } = new List<string>();

            public Dictionary<ShowInfo, List<Episode>> EpisodesNotFound { get; set; } = new();
            public Dictionary<string, ELanguage> SeasonStructureIssues { get; internal set; } = new();
        }
    }


}
