﻿using SerienHelper.API.Interfaces;
using System.Net;
using System.Text.RegularExpressions;

namespace SerienHelper.API.Handling
{
	public class BSHandler : BaseWebsiteHandler, IHandler
    {
        string baseUrl = "https://bs.to";

        public override ShowInfo GetShowInfo(string url, bool getEpisodeDesc = true)
        {
            var client = new WebClient();
            client.Headers.Add("Accept-Language", "de-de,de;q=0.8,en-us;q=0.5,en;q=0.3");
            var html = client.DownloadString(url);

            var languagesSelectMatch = Regex.Match(html, "<select class=\"series-language\">(.*?)</select>", RegexOptions.Singleline);
            var languageMatches = Regex.Matches(languagesSelectMatch.Value, "<option value=\"(.*?)\"(.*?)>(.*?)</option>");

            var showInfo = GetShowInfoFromHtml(html, url, getEpisodeDesc);

            foreach (Match languageMatch in languageMatches)
            {
                if (string.IsNullOrWhiteSpace(languageMatch.Groups[2].Value))
                {
                    var languageUrl = url + "/" + languageMatch.Groups[1].Value;
                    html = client.DownloadString(languageUrl);
                    var showInfo2 = GetShowInfoFromHtml(html, url, getEpisodeDesc);

                    foreach (var otherSeason in showInfo2.Seasons)
                    {
                        var season = showInfo.Seasons.FirstOrDefault(x => x.Nr == otherSeason.Nr);

                        if (season != null)
                        {
                            foreach (var otherEpisode in otherSeason.Episodes)
                            {
                                if (season.Episodes.Any(x => x == null))
                                {
                                    season.ToString();
                                }


                                var episode = season.Episodes.FirstOrDefault(x => x.Nr == otherEpisode.Nr);
                                if (episode != null)
                                {
                                    episode.Languages.AddRange(otherEpisode.Languages);
                                }
                                else
                                {
                                    season.Episodes.Add(otherEpisode);
                                }
                            }
                        }
                        else
                        {
                            showInfo.Seasons.Add(otherSeason);
                        }
                    }
                }
            }

            showInfo.Seasons.ForEach(x =>
            {
                x.Url = RemoveUrlEnd(x.Url);
            });


            return showInfo;
        }

        public override string GetShowInfoFilename()
        {
            return "info_bs.json";
        }

        public override ShowInfo GetShowInfoFromHtml(string html, string url, bool getEpisodeDesc = true)
        {
            var sectionMatch = Regex.Match(html, "<section class=\"serie\">(.*?)</section", RegexOptions.Singleline);
            var section = sectionMatch.Groups[1].Value;

            var titleMatch = Regex.Match(section, "<h2>(.*?)<", RegexOptions.Singleline);
            var descriptionMatch = Regex.Match(section, "<p>(.*?)</p>", RegexOptions.Singleline);

            var genresMatch = Regex.Match(section, "<span>Genres</span>.*?<p>(.*?)</p>", RegexOptions.Singleline);
            var genreMatch = Regex.Matches(genresMatch.Groups[1].Value, "<span.*?>(.*?)</span>");

            var productionTimeMatch = Regex.Match(section, "<span>Produktionsjahre</span>.*?<p>(.*?)</p>", RegexOptions.Singleline);
            var productionTimesMatch = Regex.Match(productionTimeMatch.Groups[1].Value, @".*?<em>(\d\d\d\d).-.(\d\d\d\d)</em>");

            var artistsMatch = Regex.Match(section, "<span>Hauptdarsteller</span>.*?<p>(.*?)</p>", RegexOptions.Singleline);
            var artistMatch = Regex.Matches(artistsMatch.Groups[1].Value, "<span.*?>(.*?)</span>");

            var producersMatch = Regex.Match(section, "<span>Produzenten</span>.*?<p>(.*?)</p>", RegexOptions.Singleline);
            var producerMatch = Regex.Matches(producersMatch.Groups[1].Value, "<span.*?>(.*?)</span>");

            var regieMatch = Regex.Match(section, "<span>Regisseure</span>.*?<p>(.*?)</p>", RegexOptions.Singleline);
            var regiesMatch = Regex.Matches(regieMatch.Groups[1].Value, "<span.*?>(.*?)</span>");

            var authorMatch = Regex.Match(section, "<span>Autoren</span>.*?<p>(.*?)</p>", RegexOptions.Singleline);
            var authorsMatch = Regex.Matches(authorMatch.Groups[1].Value, "<span.*?>(.*?)</span>");

            var imageUrlMatch = Regex.Match(section, "<img src=\"(.*?)\" alt=\"Cover\"", RegexOptions.Singleline);


            var showInfo = new ShowInfo()
            {
                Title = titleMatch.Groups[1].Value.Trim(),
                Description = descriptionMatch.Groups[1].Value,
                Url = url,
                ImageUrl = baseUrl + imageUrlMatch.Groups[1].Value,
                ProductionStart = productionTimesMatch.Groups[1].Value,
                ProductionEnd = productionTimesMatch.Groups[2].Value,
                Regie = regiesMatch.Select(x => x.Groups[1].Value).ToList(),
                Artists = artistMatch.Select(x => x.Groups[1].Value).ToList(),
                Producer = producerMatch.Select(x => x.Groups[1].Value).ToList(),
                Genre = genreMatch.Select(x => x.Groups[1].Value).ToList(),
                Authors = authorsMatch.Select(x => x.Groups[1].Value).ToList(),
            };

            var seasonsBlockMatch = Regex.Match(section, "<div class=\"seasons\">(.*?)<div class=\"scrollbar\">", RegexOptions.Singleline);
            var seasonsMatch = Regex.Matches(seasonsBlockMatch.Value, "<a href=\"(.*?)\">(.*?)</a>");

            showInfo.Seasons = MatchSeasons(html);

            return showInfo;
        }

        private List<Season> MatchSeasons(string html)
        {
            var sectionMatch = Regex.Match(html, "<section class=\"serie\">(.*?)</section", RegexOptions.Singleline);
            var section = sectionMatch.Groups[1].Value;

            var seasonsBlockMatch = Regex.Match(section, "<div class=\"seasons\">(.*?)<div class=\"scrollbar\">", RegexOptions.Singleline);
            var seasonsMatch = Regex.Matches(seasonsBlockMatch.Value, "<a href=\"(.*?)\">(.*?)</a>");

            int seasonNr;
            var seasons = seasonsMatch.Select(x => new Season()
            {
                Url = baseUrl + "/" + x.Groups[1].Value,
                Title = x.Groups[2].Value,
                Nr = int.TryParse(x.Groups[2].Value, out seasonNr) ? seasonNr : 0,
                SeasonType = int.TryParse(x.Groups[2].Value, out seasonNr) ? Season.ESeasonType.Season : Season.ESeasonType.Special,

            }).ToList();

            var client = new WebClient();
            client.Headers.Add("Accept-Language", "de-de,de;q=0.8,en-us;q=0.5,en;q=0.3");

            foreach (var season in seasons)
            {
                var seasonHtml = client.DownloadString(season.Url);
                var episodeTableMatch = Regex.Match(seasonHtml, "<table class=\"episodes\">(.*?)</table>", RegexOptions.Singleline);
                var episodesMatch = Regex.Matches(episodeTableMatch.Value, "<tr(.*?)<a href=\"(.*?)\" title=\"(.*?)\">(.*?)</a>.*?</tr>", RegexOptions.Singleline);

                var languagesSelectMatch = Regex.Match(html, "<select class=\"series-language\">(.*?)</select>", RegexOptions.Singleline);
                var languageMatches = Regex.Matches(languagesSelectMatch.Value, "<option value=\"(.*?)\"(.*?)>(.*?)</option>");

                var languageMatch = languageMatches.FirstOrDefault(x => !string.IsNullOrWhiteSpace(x.Groups[2].Value));
                var language = ELanguage.Unknown;
                if (languageMatch != null)
                {
                    var languageStr = languageMatch.Groups[1].Value;

                    switch (languageStr)
                    {
                        case "de":
                            language = ELanguage.Deutsch;
                            break;
                        case "en":
                            language = ELanguage.Englisch;
                            break;
                        case "des":
                            language = ELanguage.Englisch_Deutsch;
                            break;
                    }
                }

                int episodeNr;
                foreach (Match episodeMatch in episodesMatch)
                {
                    if (episodeMatch.Groups[1].Value.Contains("disabled")) continue;
                    var uri = RemoveUrlEnd(baseUrl + "/" + episodeMatch.Groups[2].Value);
                    season.Episodes.Add(new Episode()
                    {
                        Url = uri,
                        Title = episodeMatch.Groups[3].Value,
                        Nr = int.TryParse(episodeMatch.Groups[4].Value, out episodeNr) ? episodeNr : 0,
                        Languages = new List<ELanguage>() { language },
                        Season = season,
                    });
                }
            }

            return seasons;
        }

        private string RemoveUrlEnd(string url)
        {
            var uri = new Uri(url);
            return uri.AbsoluteUri.Remove(uri.AbsoluteUri.Length - uri.Segments.Last().Length);
        }
    }
}
